/******************************************************************
 *FILE: atmel_mxt_intr.c
 *SW-COMPONENT: QNX atmel touch driver
 *DESCRIPTION: This source file is to configure GPIO and interrupt
 *for atmel touch device.
 *COPYRIGHT: © 2019 Robert Bosch GmbH
 *
 *This program is free software; you can redistribute  it and/or modify it
 *under  the terms of  the GNU General  Public License as published by the
 *Free Software Foundation;  either version 2 of the  License, or (at your
 *option) any later version.
 ******************************************************************/
#include <string.h>
#include <errno.h>
#ifndef UNITTEST
#include <sys/neutrino.h>
#include <input/mtouch_log.h>
#include <gpio_client.h>
#else
#include "touchscreen_header.h"
#endif
#include "himax_platform.h"

unsigned short himax_gpio_conf(himax_dev_t *himax)
{
    uint32_t gpio_number = himax->tp_intr_gpio;
    uint32_t gpio_number_tlmm;
    uint32_t gpio_config;
    int fd;

    if (-1 == (fd = gpio_open(NULL))) {
        mtouch_error(himax->log_name,"gpio_open() failed");
        error_memory("Himax_Touch: gpio_open() failed");
        goto exit_err;
    }

    if(gpio_number  <= 0 ){
        mtouch_error(himax->log_name,"Bad GPIO input param gpio=%d \n",gpio_number);
        error_memory("Himax_Touch: Bad GPIO input param gpio=%d \n",gpio_number);
        goto exit_err;
    }

    gpio_number_tlmm = gpio_number & 0xFFFF; // for TLMM upper 16 bit mask is 0s
    gpio_config = GPIO_PIN_CFG(GPIO_INPUT,GPIO_PULL_UP,GPIO_STRENGTH_2MA,0);

    /* set gpio configuration for the gpio */
    if (GPIO_SUCCESS != gpio_set_config(fd, gpio_number_tlmm, 0, gpio_config)){
        mtouch_error(himax->log_name,"gpio_set_config failed for gpio gpio_number %d \n", gpio_number);
        error_memory("Himax_Touch: gpio_set_config failed for gpio gpio_number %d \n", gpio_number);
        goto exit_err;
    }

    /* set interrupt cfg for the gpio */
    if (GPIO_SUCCESS != gpio_set_interrupt_cfg(fd, gpio_number_tlmm, GPIO_INTERRUPT_TRIGGER_LOW, NULL)){
        mtouch_error(himax->log_name,"gpio_set_config failed for gpio gpio_number %d \n", gpio_number);
        error_memory("Himax_Touch: gpio_set_config failed for gpio gpio_number %d \n", gpio_number);
        goto exit_err;
    }

    /* get interrupt irq number from the configured interrupt gpio */
    if (GPIO_SUCCESS != gpio_get_interrupt_cfg(fd, gpio_number_tlmm, &himax->tp_intr)){
        mtouch_error(himax->log_name," failed to get irq for gpio gpio_number %d \n", gpio_number);
        error_memory("Himax_Touch: failed to get irq for gpio gpio_number %d \n", gpio_number);
        goto exit_err;
    }

    mtouch_info(himax->log_name," irq corresponding to gpio %d is %d - exiting\n", gpio_number, himax->tp_intr);

    return 0;
exit_err:
    return -1;
}

int
himax_tp_intr_attach(himax_dev_t *himax)
{
    int ret;

    ret = himax_gpio_conf(himax);
    if (ret == -1) {
        mtouch_error(himax->log_name,"faled to set gpio configuration \n ");
        error_memory("Himax_Touch: faled to set gpio configuration \n ");
        goto fail1;
    }

    SIGEV_PULSE_INIT(&himax->tp_intrevent, himax->thread_coid, HIMAX_PULSE_PRIORITY, HIMAX_PULSE_CODE, 0);

    himax->tp_iid = InterruptAttachEvent(himax->tp_intr, &himax->tp_intrevent, _NTO_INTR_FLAGS_TRK_MSK | _NTO_INTR_FLAGS_PROCESS);
    if (himax->tp_iid == -1) {
        mtouch_error(himax->log_name, "InterruptAttach: %s", strerror(errno));
        error_memory("Himax_Touch: InterruptAttach: %s", strerror(errno));
        goto fail1;
    }

    mtouch_info(himax->log_name, "Attached to device with interrupt %d", himax->tp_intr);

    return 0;

fail1:
    return -1;
}

#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn.ott.qnx.com/product/branches/7.0.0/trunk/hardware/mtouch/atmel/atmel_tp_intr.c $ $Rev: 809524 $")
#endif
